<?php
/**
* Authorize.net CIM implementation
* Customer Information Manager
* D I S C L A I M E R                                                                                          
* WARNING: ANY USE BY YOU OF THE SAMPLE CODE PROVIDED IS AT YOUR OWN RISK.                                                                                   
* Authorize.Net provides this code "as is" without warranty of any kind, either express or implied,
* including but not limited to the implied warranties of merchantability and/or fitness for a particular purpose.   
* Authorize.Net owns and retains all right, title and interest in and to the Automated Recurring Billing intellectual property.
* Author : Sandeep S. Gawali
* Email : sandeepgawali_273@yahoo.co.in
* Reference URL : http://www.authorize.net/support/CIM_XML_guide.pdf
*/

class AuthorizeNetCIM extends CApplicationComponent
{

	// set the loginname, transactionkey,  mode and host
    public $config = array();
    
    /**
     * @var string Authorize.net provided loginname
     */    
    public $cim_login_name;

    /**
     * @var string Authorize.net provided transactionkey
     */    
    public $cim_trans_key;

    /**
     * @var string Authorize.net mode liveMode / testMode
     */    
    public $cim_mode;

    /**
     * @var string Authorize.net host api.authorize.net / apitest.authorize.net
     */    
    public $cim_host;
	
	/**
     * @var string Authorize.net path
     */    
    public $cim_apipath = '/xml/v1/request.api';
	
	/**
	 * Returns settings declared in the configuraion classes.
	 * For perfomance reasons it uses Yii::app()->cache to store settings array.
	 * @return array services settings.
	 */
	public function getConfig() {
		if (Yii::app()->hasComponent('cache'))
			$config = Yii::app()->cache->get('AuthorizeNetCIM.config');
		if (!isset($config) || !is_array($config)) {
			$config = array();
			foreach ($this->config as $configElem => $value) {
				$config[$configElem] = $value;
			}
			if (Yii::app()->hasComponent('cache'))
				Yii::app()->cache->set('AuthorizeNetCIM.config', $config);
		}
		return $config;
	}
	
	/**
	* get common block
	*/
	public function getMerchantAuthenticationBlock() {
		if (!$this->cim_login_name)
                    throw new AuthorizeNetCIMException('AuthorizeNetCIM Login Name not specified.');
		if (!$this->cim_trans_key)
                    throw new AuthorizeNetCIMException('AuthorizeNetCIM Transaction Key not specified.');
		//if (!$this->cim_mode)
        //            throw new AuthorizeNetCIMException('AuthorizeNetCIM Mode not specified.');
					
		return
			"<merchantAuthentication>".
			"<name>" . CHtml::encode(strip_tags($this->cim_login_name)) . "</name>".
			"<transactionKey>" . CHtml::encode(strip_tags($this->cim_trans_key)) . "</transactionKey>".
			"</merchantAuthentication>";		
	
	}
	
	/**
	* function to send xml request to Api.
	* There is more than one way to send https requests in PHP.
	*/
	public function sendXMLRequest($content)
	{
		if (!$this->cim_host)
                    throw new AuthorizeNetCIMException('AuthorizeNetCIM Host not specified.');
		if (!$this->cim_apipath)
                    throw new AuthorizeNetCIMException('AuthorizeNetCIM API Path not specified.');
		
		$g_apihost = $this->cim_host;
		$g_apipath = $this->cim_apipath;
		
		return $this->sendRequestViaFsockopen($g_apihost, $g_apipath, $content);
	}
	
	/**
	* function to send xml request via fsockopen
	* It is a good idea to check the http status code.	
	*/
	public function sendRequestViaFsockopen($host,$path,$content)
	{
		$posturl = "ssl://" . $host;
		$header = "Host: $host\r\n";
		$header .= "User-Agent: PHP Script\r\n";
		$header .= "Content-Type: text/xml\r\n";
		$header .= "Content-Length: ".strlen($content)."\r\n";
		$header .= "Connection: close\r\n\r\n";
		$fp = fsockopen($posturl, 443, $errno, $errstr, 30);
		if (!$fp)
		{
			$body = false;
		}
		else
		{
			error_reporting(E_ERROR);
			fputs($fp, "POST $path  HTTP/1.1\r\n");
			fputs($fp, $header.$content);
			fwrite($fp, $out);
			$response = "";
			while (!feof($fp))
			{
				$response = $response . fgets($fp, 128);
			}
			fclose($fp);
			error_reporting(E_ALL ^ E_NOTICE);
			
			$len = strlen($response);
			$bodypos = strpos($response, "\r\n\r\n");
			if ($bodypos <= 0)
			{
				$bodypos = strpos($response, "\n\n");
			}
			while ($bodypos < $len && $response[$bodypos] != '<')
			{
				$bodypos++;
			}
			$body = substr($response, $bodypos);
		}
		return $body;
	}
	
	/**
	* function to parse the api response
	* The code uses SimpleXML. http://us.php.net/manual/en/book.simplexml.php 
	* There are also other ways to parse xml in PHP depending on the version and what is installed.
	*/
	public function parseAPIResponse($content)
	{
		$parsedresponse = simplexml_load_string($content, "SimpleXMLElement", LIBXML_NOWARNING);
		/*if ("Ok" != $parsedresponse->messages->resultCode) {
			echo "The operation failed with the following errors:<br>";
			foreach ($parsedresponse->messages->message as $msg) {
				echo "[" . htmlspecialchars($msg->code) . "] " . htmlspecialchars($msg->text) . "<br>";
			}
			echo "<br>";
		}*/
		return $parsedresponse;
	}	
		
	
	/**
	* create profile...
	* How to use
	* $data = array();
	* $data = array(
	*				'type' => 'Payment',              //Payment OR Withdraw
	*				'user_id' => '101',
	*				'profile_id' => '202',
	*				'email' => 'test@emailid.com'
	*			);
	* $result = Yii::app()->authorizenetCIM->authNetCreateProfile($data);
	*
	* if($result != "error"){ 'SUCCESS' }else{ 'ERROR' }
	*/
	public function authNetCreateProfile($data) //$type = Payment OR Withdraw
	{
		//build xml to post
		$content =
			"<?xml version=\"1.0\" encoding=\"utf-8\"?>" .
			"<createCustomerProfileRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">" .
			$this->getMerchantAuthenticationBlock().
			"<profile>".
			"<merchantCustomerId>" . CHtml::encode(strip_tags($data['type']))."-".CHtml::encode(strip_tags($data['user_id']))."-".CHtml::encode(strip_tags($data['profile_id']))  ."</merchantCustomerId>". // Your own identifier for the customer.
			"<description></description>".
			"<email>" . CHtml::encode(strip_tags($data['email'])) . "</email>".
			"</profile>".
			"</createCustomerProfileRequest>";	
		
		$response = $this->sendXMLRequest($content);	
		$parsedresponse = $this->parseAPIResponse($response);	
		
		if ("Ok" == $parsedresponse->messages->resultCode) {
			return htmlspecialchars($parsedresponse->customerProfileId);
		}	
		
		return "error";
	}
	
	/**
	* delete profile...
	* How to use
	* $result = Yii::app()->authorizenetCIM->authNetDeleteProfile(123456);
	*
	* if($result['result'] == 'success'){ 'SUCCESS' }else{ 'ERROR' }
	*/
	function authNetDeleteProfile($customerProfileId)
	{
		//build xml to post
		$content =
			"<?xml version=\"1.0\" encoding=\"utf-8\"?>" .
			"<deleteCustomerProfileRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">" .
			$this->getMerchantAuthenticationBlock().
			"<customerProfileId>" . CHtml::encode(strip_tags($customerProfileId)) . "</customerProfileId>".
			"</deleteCustomerProfileRequest>";	
		
		$response = $this->sendXMLRequest($content);	
		$parsedresponse = $this->parseAPIResponse($response);

		if ("Ok" == $parsedresponse->messages->resultCode) {
			return array("result"=>"success", "succmessage"=>"customerProfileId ".htmlspecialchars($customerProfileId)." was successfully deleted.");
		}
			
		return array("result"=>"error", "errmessage"=>htmlspecialchars($parsedresponse->messages->message->text));
	}
	
	
	/**
	* create payment profile...
	* How to use
	* $data = array();
	* $data = array(
	*				'type' => 'Payment',              //Payment OR Withdraw
	*				'user_id' => '101',
	*				'profile_id' => '202',
	*				'customerProfileId' => 'XXXX',
	*				'merchantCustomerId' => 'XXXX',
	*				'firstname' => 'Ryan',
	*				'lastname' => 'Smith',
	*				'card_number' => 'XXXX',
	*				'cvv_code' => 'XX',
	*				'card_expire_month' => '03', //MM
	*				'card_expire_year' => '2013', //YYYY
	*				'address' => 'address line',
	*				'city' => 'city name',
	*				'state' => 'state name', //State Code like MH - Maharastra 
	*				'zip' => 'zip code',
	*				'country' => 'IN' //Country Code like IN - India
	*			);
	* $result = Yii::app()->authorizenetCIM->authNetCreatePaymentProfile($data);
	*
	* if($result['result'] == 'success'){ 'SUCCESS' }else{ 'ERROR' }
	*/
	function authNetCreatePaymentProfile($data)
	{
		if (!$this->cim_mode)
                    throw new AuthorizeNetCIMException('AuthorizeNetCIM Mode not specified.');			

		$modeTestLive = $this->cim_mode;
		
		//no need of this code if expire month value is leading with zero if it's <= 9
		//if($data['card_expire_month'] <= 9)
		//{
		//	$data['card_expire_month'] = "0".$dataArr['card_expire_month'];
		//}
		
		//build xml to post
		$content =
			"<?xml version=\"1.0\" encoding=\"utf-8\"?>" .
			"<createCustomerPaymentProfileRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">" .
			$this->getMerchantAuthenticationBlock().
			"<customerProfileId>" . CHtml::encode(strip_tags($data['customerProfileId'])) . "</customerProfileId>".
			"<paymentProfile>".
			"<billTo>".
			 "<firstName>" . CHtml::encode(strip_tags($data['firstname'])) . "</firstName>".
			 "<lastName>" . CHtml::encode(strip_tags($data['lastname'])) . "</lastName>".
			 "<company/>".
			 "<address>" . CHtml::encode(strip_tags($data['address'])) . "</address>".
			 "<city>" . CHtml::encode(strip_tags($data['city'])) . "</city>".
			 "<state>" . CHtml::encode(strip_tags($data['state'])) . "</state>".
			 "<zip>" . CHtml::encode(strip_tags($data['zip'])) . "</zip>".
			 "<country>" . CHtml::encode(strip_tags($data['country'])) . "</country>".			 
			 //"<phoneNumber>000-000-0000</phoneNumber>".		 
			 //"<faxNumber>000-000-0000</faxNumber>".
			"</billTo>".
			"<payment>".
			 "<creditCard>".
			  "<cardNumber>" . CHtml::encode(strip_tags($data['card_number'])) . "</cardNumber>".
			  "<expirationDate>" . CHtml::encode(strip_tags($data['card_expire_year']))."-".CHtml::encode(strip_tags($data['card_expire_month'])) . "</expirationDate>". // required format for API is YYYY-MM
			  "<cardCode>" . CHtml::encode(strip_tags($data['cvv_code'])) . "</cardCode>".
			 "</creditCard>".
			"</payment>".
			"</paymentProfile>".
			"<validationMode>" . CHtml::encode(strip_tags($modeTestLive)) . "</validationMode>". // or testMode or liveMode
			"</createCustomerPaymentProfileRequest>";	
		
		$response = $this->sendXMLRequest($content);	
		$parsedresponse = $this->parseAPIResponse($response);

		if ("Ok" == $parsedresponse->messages->resultCode) {
			return array("result"=>"success", "customerPaymentProfileId"=>htmlspecialchars($parsedresponse->customerPaymentProfileId));
		}	
		
		return array("result"=>"error", "errmessage"=>htmlspecialchars($parsedresponse->messages->message->text));
	}
	
	/**
	* delete payment profile...
	* How to use
	* $result = Yii::app()->authorizenetCIM->authNetDeletePaymentProfile($customerProfileId, $customerPaymentProfileId);
	*
	* if($result['result'] == 'success'){ 'SUCCESS' }else{ 'ERROR' }
	*/
	function authNetDeletePaymentProfile($customerProfileId, $customerPaymentProfileId)
	{
		//build xml to post
		$content =
			"<?xml version=\"1.0\" encoding=\"utf-8\"?>" .
			"<deleteCustomerPaymentProfileRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">" .
			$this->getMerchantAuthenticationBlock().
			"<customerProfileId>" . CHtml::encode(strip_tags($customerProfileId)) . "</customerProfileId>".
			"<customerPaymentProfileId>" . CHtml::encode(strip_tags($customerPaymentProfileId)) . "</customerPaymentProfileId>".
			"</deleteCustomerPaymentProfileRequest>";	
		
		$response = $this->sendXMLRequest($content);	
		$parsedresponse = $this->parseAPIResponse($response);
		
		if ("Ok" == $parsedresponse->messages->resultCode) {
			return array("result"=>"success", "succmessage"=>"customerPaymentProfileId ".htmlspecialchars($customerPaymentProfileId)." was successfully deleted.");
		}
			
		return array("result"=>"error", "errmessage"=>htmlspecialchars($parsedresponse->messages->message->text));
	}	
	
	/**
	* get customer payment profile...
	* How to use
	* $result = Yii::app()->authorizenetCIM->authNetGetCustomerPaymentProfile($customerProfileId, $customerPaymentProfileId);
	*
	* if($result){ 'SUCCESS' }else{ 'ERROR' }
	*/
	function authNetGetCustomerPaymentProfile($customerProfileId, $customerPaymentProfileId)
	{
		if($customerProfileId > 0 && $customerPaymentProfileId > 0)
		{
			//build xml to post
			$content =
				"<?xml version=\"1.0\" encoding=\"utf-8\"?>" .
				"<getCustomerPaymentProfileRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">" .
				$this->getMerchantAuthenticationBlock().
				"<customerProfileId>" . CHtml::encode(strip_tags($customerProfileId)) . "</customerProfileId>" .
				"<customerPaymentProfileId>" . CHtml::encode(strip_tags($customerPaymentProfileId)) . "</customerPaymentProfileId>" .
				"</getCustomerPaymentProfileRequest>";	
		
				$response = $this->sendXMLRequest($content);	
				$parsedresponse = $this->parseAPIResponse($response);
				
				if ("Ok" == $parsedresponse->messages->resultCode) {
					$responseArray = array();
					$cc_firstName = htmlspecialchars($parsedresponse->paymentProfile->billTo->firstName);
					$cc_lastName = htmlspecialchars($parsedresponse->paymentProfile->billTo->lastName);
					$cc_customerPaymentProfileId = htmlspecialchars($parsedresponse->paymentProfile->customerPaymentProfileId);
					$cc_cardNumber = htmlspecialchars($parsedresponse->paymentProfile->payment->creditCard->cardNumber);
					$cc_expirationDate = htmlspecialchars($parsedresponse->paymentProfile->payment->creditCard->expirationDate);
					
					$responseArray = array(
											"firstName"=>$cc_firstName,
											"lastName"=>$cc_lastName,
											"customerPaymentProfileId"=>$cc_customerPaymentProfileId,
											"cardNumber"=>$cc_cardNumber,
											"expirationDate"=>$cc_expirationDate
										);						
					return $responseArray;
					
				}
				return false;
				
		}
		return false;
	}
	
	/**
	* create transaction...
	* How to use
	* $data = array();
	* $data = array(
	*				'customerProfileId' => 'XXXX',
	*				'customerPaymentProfileId' => 'XXXX',
	*				'customerShippingAddressId' => '',
	*				'itemId' => time(),
	*				'name' => 'Payment Credit Card',
	*				'description' => 'Payment Credit Card Description',
	*				'amount' => '1.10'
	*			);
	* $result = Yii::app()->authorizenetCIM->authNetCreateTransaction($data);
	*
	* if($result['result'] == "success" && $result['responseCode'] == "1"){ 'SUCCESS' }else{ 'ERROR' }
	*/
	function authNetCreateTransaction($data)
	{
		//build xml to post
		$content =
			"<?xml version=\"1.0\" encoding=\"utf-8\"?>" .
			"<createCustomerProfileTransactionRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">" .
			$this->getMerchantAuthenticationBlock().
			"<transaction>".
			"<profileTransAuthCapture>".
			"<amount>" . CHtml::encode(strip_tags($data["amount"])) . "</amount>". // should include tax, shipping, and everything.
			/*"<shipping>".
			"<amount>0.00</amount>".
			"<name>Free Shipping</name>".
			"<description>Free UPS Ground shipping. Ships in 5-10 days.</description>".
			"</shipping>".*/
			"<lineItems>".
			"<itemId>" . CHtml::encode(strip_tags($data["itemId"])) . "</itemId>".
			"<name>" . CHtml::encode(strip_tags($data["name"])) . "</name>".
			"<description>" . CHtml::encode(strip_tags($data["description"])) . "</description>".
			"<quantity>1</quantity>".
			"<unitPrice>" . CHtml::encode(strip_tags($data["amount"])) . "</unitPrice>".
			"<taxable>false</taxable>".
			"</lineItems>".
			"<customerProfileId>" . CHtml::encode(strip_tags($data["customerProfileId"])) . "</customerProfileId>".
			"<customerPaymentProfileId>" . CHtml::encode(strip_tags($data["customerPaymentProfileId"])) . "</customerPaymentProfileId>".
			//"<customerShippingAddressId>" . CHtml::encode(strip_tags($data["customerShippingAddressId"])) . "</customerShippingAddressId>".
			"<order>".
			"<invoiceNumber>INV-".time()."</invoiceNumber>".
			"</order>".
			"</profileTransAuthCapture>".
			"</transaction>".
			"</createCustomerProfileTransactionRequest>";	
		
			$response = $this->sendXMLRequest($content);	
			$parsedresponse = $this->parseAPIResponse($response);
			
			$responseArray = array();
			
			if ("Ok" == $parsedresponse->messages->resultCode) {
										
					if (isset($parsedresponse->directResponse)) {
					
						$directResponseFields = explode(",", $parsedresponse->directResponse);
						$responseCode = htmlspecialchars($directResponseFields[0]); // 1 = Approved 2 = Declined 3 = Error
						$responseReasonCode = htmlspecialchars($directResponseFields[2]); // See http://www.authorize.net/support/AIM_guide.pdf
						$responseReasonText = htmlspecialchars($directResponseFields[3]);
						$approvalCode = htmlspecialchars($directResponseFields[4]); // Authorization code
						$transId = htmlspecialchars($directResponseFields[6]);
						
						//if ("1" == $responseCode) echo "The transaction was successful.<br>";
						//else if ("2" == $responseCode) echo "The transaction was declined.<br>";
						//else echo "The transaction resulted in an error.<br>";
						
						$responseArray = array(
												"result"=>"success",
												"message"=>"A transaction was successfully created.",
												"responseCode"=>$responseCode,
												"responseReasonCode"=>$responseReasonCode,
												"responseReasonText"=>$responseReasonText,
												"approvalCode"=>$approvalCode,
												"transactionid"=>$transId
											);
					}															
					
			}else
			{
				$responseArray = array(
												"result"=>"error",
												"message"=>htmlspecialchars($parsedresponse->messages->message->text)
											);
			
			}					
			return $responseArray;
	}
	
	/**
	* create refund transaction...
	* How to use
	* $data = array();
	* $data = array(
	*				'customerProfileId' => 'XXXX',
	*				'customerPaymentProfileId' => 'XXXX',
	*				'customerShippingAddressId' => '',
	*				'itemId' => time(),
	*				'name' => 'Payment CC Refund',
	*				'description' => 'Refunded amount to the charged cc',
	*				'amount' => '1.10', //Amount same as charged amount or less than
	*				'transactionid' => 'cc charged transtion id'
	*			);
	* $result = Yii::app()->authorizenetCIM->authNetCreateRefundTransaction($data);
	*
	* if($result['result'] == "success" && $result['responseCode'] == "1"){ 'SUCCESS' }else{ 'ERROR' }
	*/
	function authNetCreateRefundTransaction($data)
	{
		$creditCardNumberMasked = "";
		$custPaymentProfileDet = $this->authNetGetCustomerPaymentProfile($data["customerProfileId"], $data["customerPaymentProfileId"]);
		$creditCardNumberMasked = $custPaymentProfileDet['cardNumber'];
		
		//build xml to post
		$content =
			"<?xml version=\"1.0\" encoding=\"utf-8\"?>" .
			"<createCustomerProfileTransactionRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">" .
			$this->getMerchantAuthenticationBlock().
			"<transaction>" .
			"<profileTransRefund>" .
			"<amount>" . CHtml::encode(strip_tags($data["amount"])) . "</amount>" . // should include tax, shipping, and everything.
			"<customerProfileId>" . CHtml::encode(strip_tags($data["customerProfileId"])) . "</customerProfileId>".
			"<customerPaymentProfileId>" . CHtml::encode(strip_tags($data["customerPaymentProfileId"])) . "</customerPaymentProfileId>".
			//"<customerShippingAddressId>" . CHtml::encode(strip_tags($data["customerShippingAddressId"])) . "</customerShippingAddressId>"..
			"<creditCardNumberMasked>" . CHtml::encode(strip_tags($creditCardNumberMasked)) . "</creditCardNumberMasked>" .
			"<order>" .
			"<invoiceNumber>INVREF-".time()."</invoiceNumber>" .
			"</order>" .
			"<transId>" . CHtml::encode(strip_tags($data["transactionid"])) . "</transId>" .
			"</profileTransRefund>" .
			"</transaction>" .
			"</createCustomerProfileTransactionRequest>";	
		
			$response = $this->sendXMLRequest($content);	
			$parsedresponse = $this->parseAPIResponse($response);
			
			$responseArray = array();
			
			if ("Ok" == $parsedresponse->messages->resultCode) {
										
					if (isset($parsedresponse->directResponse)) {
					
						$directResponseFields = explode(",", $parsedresponse->directResponse);
						$responseCode = htmlspecialchars($directResponseFields[0]); // 1 = Approved 2 = Declined 3 = Error
						$responseReasonCode = htmlspecialchars($directResponseFields[2]); // See http://www.authorize.net/support/AIM_guide.pdf
						$responseReasonText = htmlspecialchars($directResponseFields[3]);
						$approvalCode = htmlspecialchars($directResponseFields[4]); // Authorization code
						$transId = htmlspecialchars($directResponseFields[6]);
						
						//if ("1" == $responseCode) echo "The transaction was successful.<br>";
						//else if ("2" == $responseCode) echo "The transaction was declined.<br>";
						//else echo "The transaction resulted in an error.<br>";
						
						$responseArray = array(
												"result"=>"success",
												"messageid"=>htmlspecialchars($parsedresponse->messages->message->code),
												"message"=>"A refund transaction was successfully created.",
												"responseCode"=>$responseCode,
												"responseReasonCode"=>$responseReasonCode,
												"responseReasonText"=>$responseReasonText,
												"approvalCode"=>$approvalCode,
												"transactionid"=>$transId
											);
					}															
					
			}else
			{
				$responseArray = array(
												"result"=>"error",
												"messageid"=>htmlspecialchars($parsedresponse->messages->message->code),
												"message"=>htmlspecialchars($parsedresponse->messages->message->text)
											);
			
			}
						
			return $responseArray;
	}
	
	/**
	* create void transaction...
	* How to use
	* $data = array();
	* $data = array(
	*				'customerProfileId' => 'XXXX',
	*				'customerPaymentProfileId' => 'XXXX',
	*				'customerShippingAddressId' => '',
	*				'transactionid' => 'cc charged transtion id'
	*			);
	* $result = Yii::app()->authorizenetCIM->authNetCreateVoidTransaction($data);
	*
	* if($result['result'] == "success" && $result['responseCode'] == "1"){ 'SUCCESS' }else{ 'ERROR' }
	*/
	function authNetCreateVoidTransaction($data)
	{
		//build xml to post
		$content =
			"<?xml version=\"1.0\" encoding=\"utf-8\"?>" .
			"<createCustomerProfileTransactionRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">".
			$this->getMerchantAuthenticationBlock().
			"<transaction>".
			"<profileTransVoid>".
			"<customerProfileId>" . CHtml::encode(strip_tags($data["customerProfileId"])) . "</customerProfileId>".
			"<customerPaymentProfileId>" . CHtml::encode(strip_tags($data["customerPaymentProfileId"])) . "</customerPaymentProfileId>".
			//"<customerShippingAddressId>" . CHtml::encode(strip_tags($data["customerShippingAddressId"])) . "</customerShippingAddressId>".
			"<transId>" . CHtml::encode(strip_tags($data["transactionid"])) . "</transId>".
			"</profileTransVoid>".
			"</transaction>".
			"</createCustomerProfileTransactionRequest>";	
		
			$response = $this->sendXMLRequest($content);	
			$parsedresponse = $this->parseAPIResponse($response);
			
			$responseArray = array();
			
			if ("Ok" == $parsedresponse->messages->resultCode) {
										
					if (isset($parsedresponse->directResponse)) {
					
						$directResponseFields = explode(",", $parsedresponse->directResponse);
						$responseCode = htmlspecialchars($directResponseFields[0]); // 1 = Approved 2 = Declined 3 = Error
						$responseReasonCode = htmlspecialchars($directResponseFields[2]); // See http://www.authorize.net/support/AIM_guide.pdf
						$responseReasonText = htmlspecialchars($directResponseFields[3]);
						$approvalCode = htmlspecialchars($directResponseFields[4]); // Authorization code
						$transId = htmlspecialchars($directResponseFields[6]);
						
						//if ("1" == $responseCode) echo "The transaction was successful.<br>";
						//else if ("2" == $responseCode) echo "The transaction was declined.<br>";
						//else echo "The transaction resulted in an error.<br>";
						
						$responseArray = array(
												"result"=>"success",
												"message"=>"A transaction was successfully created.",
												"responseCode"=>$responseCode,
												"responseReasonCode"=>$responseReasonCode,
												"responseReasonText"=>$responseReasonText,
												"approvalCode"=>$approvalCode,
												"transactionid"=>$transId
											);
					}															
					
			}else
			{
				$responseArray = array(
												"result"=>"error",
												"message"=>htmlspecialchars($parsedresponse->messages->message->text)
											);
			
			}				
			return $responseArray;							
	}	
	
} //end class Profile_AuthorizeNet_Model

?>
